`timescale 1ns / 1ps

module tb_top;
   
   parameter CLK_WIDTH             = 3;      // # of clock outputs
   parameter CS_NUM                = 1;      // # of separate memory chip selects
   parameter CS_WIDTH              = 1;      // # of total memory chip selects
   parameter DQ_WIDTH              = 64;      // # of data width
   parameter DQS_WIDTH             = 8;      // # of DQS strobes
   parameter ODT_WIDTH             = 1;      // # of memory on-die term enables

   parameter CLK_PERIOD            = 5000;   // Core/Mem clk period (in ps)
   localparam real CLK_PERIOD_NS   = CLK_PERIOD / 1000.0;
   localparam real TCYC_200           = 5.0;
   parameter RST_ACT_LOW           = 1;      // =1 for active low reset, =0 for active high
   localparam real TPROP_DQS          = 0.01;  // Delay for DQS signal during Write Operation
   localparam real TPROP_DQS_RD       = 0.01;  // Delay for DQS signal during Read Operation
   localparam real TPROP_PCB_CTRL     = 0.01;  // Delay for Address and Ctrl signals
   localparam real TPROP_PCB_DATA     = 0.01;  // Delay for data signal during Write operation
   localparam real TPROP_PCB_DATA_RD  = 0.01;  // Delay for data signal during Read operation



   genvar 			 j;
   genvar			 i;
   reg                           clk_in;
   wire                          clk_in_n;
   wire                          clk_in_p;
   reg                           sys_clk200;
   wire                          clk200_n;
   wire                          clk200_p;
   reg                           sys_rst_n;
   wire                          sys_rst_out;
    
   wire [DQ_WIDTH-1:0]          ddr2_dq_sdram;
   wire [DQS_WIDTH-1:0]         ddr2_dqs_sdram;
   wire [DQS_WIDTH-1:0]         ddr2_dqs_n_sdram;
   wire [7:0]                   ddr2_dm_sdram;
   reg  [7:0]           	ddr2_dm_sdram_tmp;
   reg      		        ddr2_clk_sdram;
   reg           		ddr2_clk_n_sdram;
   reg  [12:0]          	ddr2_address_sdram;
   reg  [1:0]         		ddr2_ba_sdram;
   reg                          ddr2_ras_n_sdram;
   reg                          ddr2_cas_n_sdram;
   reg                          ddr2_we_n_sdram;
   reg [CS_WIDTH-1:0]           ddr2_cs_n_sdram;
   reg 			        ddr2_cke_sdram;
   reg [ODT_WIDTH-1:0]          ddr2_odt_sdram;

   wire [DQ_WIDTH-1:0]          ddr2_dq_fpga;
   wire [DQS_WIDTH-1:0]         ddr2_dqs_fpga;
   wire [DQS_WIDTH-1:0]         ddr2_dqs_n_fpga;
   wire [7:0]          		ddr2_dm_fpga;
   wire 		        ddr2_clk_fpga;
   wire          		ddr2_clk_n_fpga;
   wire [12:0]         		ddr2_address_fpga;
   wire [1:0]        		ddr2_ba_fpga;
   wire                         ddr2_ras_n_fpga;
   wire                         ddr2_cas_n_fpga;
   wire                         ddr2_we_n_fpga;
   wire [CS_WIDTH-1:0]          ddr2_cs_n_fpga;
   wire          		ddr2_cke_fpga;
   wire [ODT_WIDTH-1:0]         ddr2_odt_fpga;


   wire 			 stx;
   wire 			 srx;
   
   wire [21:0]                    flash_addr; 
   wire [15:0]                    flash_data;
   

   
   initial begin
 
    // Display start message
         $display("INFO: TBENCH: Starting simulation...");
          
    // Create VCD trace file
    //     $dumpfile("trace.vcd");
    //     $dumpvars();
                       
    // Run the simulation
    //      sys_clock <= 1'b1;
    //     sys_reset <= 1'b1;
    //     #1000
    //     sys_reset <= 1'b0;
         #700_000 
         $display("INFO: TBENCH: Completed simulation!");
         $finish;
                                                      
    end


   //***************************************************************************
   // Clock generation and reset
   //***************************************************************************

   initial
     clk_in = 1'b0;
   always
     clk_in = #(CLK_PERIOD_NS/2) ~clk_in;

   assign                clk_in_p = clk_in;
   assign                clk_in_n = ~clk_in;

   initial
     sys_clk200 = 1'b0;
   always
     sys_clk200 = #(TCYC_200/2) ~sys_clk200;

   assign                clk200_p = sys_clk200;
   assign                clk200_n = ~sys_clk200;

   initial begin
      sys_rst_n = 1'b0;
      #200;
      sys_rst_n = 1'b1;
   end
   assign sys_rst_out = RST_ACT_LOW ? sys_rst_n : ~sys_rst_n;

   //***************************************************************************
   // W1 module instance 
   //***************************************************************************

W1 W1_inst
(
   .clk_in (clk_in),
   .sysrst (sys_rst_out),

   // ddr3 memory interface
   .ddr3_dq (ddr2_dq_fpga),            
   .ddr3_dqs (ddr2_dqs_fpga),
   .ddr3_dqs_n (ddr2_dqs_n_fpga),
   .ddr3_ck (ddr2_clk_fpga),
   .ddr3_ck_n (ddr2_clk_n_fpga),
   .ddr3_a (ddr2_address_fpga),
   .ddr3_ba (ddr2_ba_fpga), //FIXME
   .ddr3_ras_n (ddr2_ras_n_fpga),
   .ddr3_cas_n (ddr2_cas_n_fpga),
   .ddr3_we_n (ddr2_we_n_fpga),
   .ddr3_cs_n (ddr2_cs_n_fpga),
   .ddr3_odt (ddr2_odt_fpga),
   .ddr3_ce (ddr2_cke_fpga),
   .ddr3_dm (ddr2_dm_fpga),

   // Console interface
   .srx (srx),
   .stx (stx),

   //flash interface	
   .flash_addr(flash_addr),
   .flash_data(flash_data),
   .flash_oen(flash_oen),
   .flash_wen(flash_wen),
   .flash_cen(flash_cen),
   .flash_clk(flash_clk),
   .flash_adv(flash_adv),
   .flash_rst(flash_rst)
);


   //***************************************************************************
   // FLASH module instance 
   //***************************************************************************
   

flash flash_inst
(
   .flash_addr(flash_addr),
   .flash_data(flash_data),
   .flash_oen(flash_oen),
   .flash_wen(flash_wen),
   .flash_cen(flash_cen),
   .flash_clk(flash_clk),
   .flash_adv(flash_adv),
   .flash_rst(flash_rst)
);

//DDR2 model
//

 always @( * ) begin
    ddr2_clk_sdram        <=  #(TPROP_PCB_CTRL) ddr2_clk_fpga;
    ddr2_clk_n_sdram      <=  #(TPROP_PCB_CTRL) ddr2_clk_n_fpga;
    ddr2_address_sdram    <=  #(TPROP_PCB_CTRL) ddr2_address_fpga;
    ddr2_ba_sdram         <=  #(TPROP_PCB_CTRL) ddr2_ba_fpga;
    ddr2_ras_n_sdram      <=  #(TPROP_PCB_CTRL) ddr2_ras_n_fpga;
    ddr2_cas_n_sdram      <=  #(TPROP_PCB_CTRL) ddr2_cas_n_fpga;
    ddr2_we_n_sdram       <=  #(TPROP_PCB_CTRL) ddr2_we_n_fpga;
    ddr2_cs_n_sdram       <=  #(TPROP_PCB_CTRL) ddr2_cs_n_fpga;
    ddr2_cke_sdram        <=  #(TPROP_PCB_CTRL) ddr2_cke_fpga;
    ddr2_odt_sdram        <=  #(TPROP_PCB_CTRL) ddr2_odt_fpga;
    ddr2_dm_sdram_tmp     <=  #(TPROP_PCB_DATA) ddr2_dm_fpga;//DM signal generation
  end

assign ddr2_dm_sdram = ddr2_dm_sdram_tmp;

genvar dqwd;
  generate
    for (dqwd = 0;dqwd < DQ_WIDTH;dqwd = dqwd+1) begin : dq_delay
      WireDelay #
       (
        .Delay_g     (TPROP_PCB_DATA),
        .Delay_rd    (TPROP_PCB_DATA_RD)
       )
      u_delay_dq
       (
        .A           (ddr2_dq_fpga[dqwd]),
        .B           (ddr2_dq_sdram[dqwd]),
        .reset       (sys_rst_n)
       );
    end
  endgenerate


 genvar dqswd;
  generate
    for (dqswd = 0;dqswd < DQS_WIDTH;dqswd = dqswd+1) begin : dqs_delay
      WireDelay #
       (
        .Delay_g     (TPROP_DQS),
        .Delay_rd    (TPROP_DQS_RD)
       )
      u_delay_dqs
       (
        .A           (ddr2_dqs_fpga[dqswd]),
        .B           (ddr2_dqs_sdram[dqswd]),
        .reset       (sys_rst_n)
       );

      WireDelay #
       (
        .Delay_g     (TPROP_DQS),
        .Delay_rd    (TPROP_DQS_RD)
       )
      u_delay_dqs_n
       (
        .A           (ddr2_dqs_n_fpga[dqswd]),
        .B           (ddr2_dqs_n_sdram[dqswd]),
        .reset       (sys_rst_n)
       );
    end
  endgenerate

           // if the data width is multiple of 16
              //for(j = 0; j < CS_NUM; j = j+1) begin : gen_cs
                for(i = 0; i < DQS_WIDTH/2; i = i+1) begin : gen
                   ddr2_model u_mem0
                     (
                      .ck        (ddr2_clk_sdram),
                      .ck_n      (ddr2_clk_n_sdram),
                      .cke       (ddr2_cke_sdram),
                      .cs_n      (ddr2_cs_n_sdram[CS_WIDTH*i/DQS_WIDTH]),
                      .ras_n     (ddr2_ras_n_sdram),
                      .cas_n     (ddr2_cas_n_sdram),
                      .we_n      (ddr2_we_n_sdram),
                      .dm_rdqs   (ddr2_dm_sdram[(2*(i+1))-1 : i*2]),
                      .ba        (ddr2_ba_sdram), //FIXME
                      .addr      (ddr2_address_sdram),
                      .dq        (ddr2_dq_sdram[(16*(i+1))-1 : i*16]),
                      .dqs       (ddr2_dqs_sdram[(2*(i+1))-1 : i*2]),
                      .dqs_n     (ddr2_dqs_n_sdram[(2*(i+1))-1 : i*2]),
                      .rdqs_n    (),
                      .odt       (ddr2_odt_sdram[ODT_WIDTH*i/DQS_WIDTH])
                      );
                end
              //end
endmodule

